<?php
/**
 * Price Per Variant Price Method
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Mtc\Shop\Contracts\PriceMethod;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Create a 'Per Variant' price method.
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerVariant implements PriceMethod
{
    /**
     * Store the product model
     *
     * @var null|Product
     */
    protected $product = null;

    /**
     * Store the variant model
     *
     * @var null|Variant
     */
    protected $variant = null;

    /**
     * Setup the price method with the product and variant models.
     *
     * @param Mtc\Shop\Product      $product Product this is for
     * @param Mtc\Shop\Variant|null $variant Variant model
     *
     * @return void
     */
    public function __construct(Product $product, Variant $variant = null)
    {
        $this->product = $product;
        $this->variant = $variant;
    }

    /**
     * Get the price (or lowest variant price) for a product.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float Price
     */
    public function perProduct($quantity = 1)
    {
        // Bulk pricing is built in by default.
        return PricePerVariant::where('variant_id', $this->variant->id)
            ->where('quantity', '<=', $quantity)
            ->orderBy('quantity', 'asc')
            ->first()
            ->price;
    }

    /**
     * Get the min/max price for an product.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float[] A 'min' and 'max' returned in an array.
     */
    public function getRange($quantity = 1)
    {
        $prices = PricePerVariant::where('variant_id', $this->variant->id)
            ->orderBy('price', 'asc')
            ->get();

        return [
            'min' => $prices->first(),
            'max' => $prices->last(),
        ];
    }

    /**
     * Get the price for a variant.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float Price
     */
    public function perVariant($quantity = 1)
    {
        return $this->perProduct($quantity);
    }
}
