<?php
/**
 * Shop Service Provider
 *
 * Register views and translations for the current component.
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Providers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Providers;

use Blade;
use Event;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Mtc\Shop\Basket;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Contracts\ProductRepositoryContract;
use Mtc\Shop\Http\ViewComposers\BasketComposer;
use Mtc\Shop\Order\Order;
use Mtc\Shop\Product;
use Mtc\Shop\Repositories\ProductRepository;
use Mtc\Shop\ShippingFlatRate;
use Mtc\Shop\Variant;
use View;

/**
 * Connect the component to Laravel
 *
 * @category Mtc\Shop\Providers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ShopServiceProvider extends ServiceProvider
{
    /**
     * Boot the service provider.
     *
     * @return void
     */
    public function boot()
    {
        $this->defineResources();
        $this->defineRoutes();
        $this->definePublished();
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        if (! defined('SHOP_PATH')) {
            define('SHOP_PATH', realpath(__DIR__.'/../../'));
        }

        $this->app->bind(ProductRepositoryContract::class, ProductRepository::class);

        // Load a custom basket (or default).
        $this->app->singleton(
            BasketContract::class, function ($app) {
                $class = config('shop.model', Basket::class);
                return $class::findOrNew(session('basket_id'));
            }
        );

        // Define the order model
        $this->app->bind(OrderContract::class, Order::class);

        // Register the singleton class to the necessary views
        View::composer(
            [
                'theme::layouts.*',
            ],
            BasketComposer::class
        );

        if ($this->app->runningInConsole()) {
            $this->commands([]);
        }

    }

    /**
     * Define the routes.
     *
     * @return void
     */
    protected function defineRoutes()
    {
        $this->defineRouteBindings();

        /**
         * If the routes have not been cached, we will include them in a route group
         * so that all of the routes will be conveniently registered to the given
         * controller namespace. After that we will load the Core routes file.
         */
        if (! $this->app->routesAreCached()) {
            Route::group(
                ['namespace' => 'Mtc\Shop\Http\Controllers'],
                function ($router) {
                    $path = realpath(SHOP_PATH . '/routes');
                    foreach (glob("{$path}/*.php") as $file) {
                        include $file;
                    }
                }
            );
        }
    }

    /**
     * Define the route model bindings.
     *
     * @return void
     */
    protected function defineRouteBindings()
    {
        Route::model('order', OrderContract::class);
        Route::model('flat_rate', ShippingFlatRate::class);
    }

    /**
     * Define the resources for this package.
     *
     * @return void
     */
    protected function defineResources()
    {
        $this->loadViewsFrom(SHOP_PATH . '/resources/views', 'shop');
        $this->loadTranslationsFrom(SHOP_PATH . '/resources/lang', 'shop');
        $this->mergeConfigFrom(
            __DIR__ . '/../../config/tax.php', 'tax'
        );
        $this->mergeConfigFrom(
            __DIR__ . '/../../config/stock.php', 'stock'
        );

    }

    protected function definePublished()
    {
        $this->publishes(
            [
            __DIR__ . '/../../config/' => config_path()
            ], 'config'
        );

        $this->publishes(
            [
            __DIR__ . '/../../database/migrations/' => database_path('migrations')
            ], 'migrations'
        );

        $this->publishes(
            [
            __DIR__ . '/../../database/seeds/' => database_path('seeds')
            ], 'seeds'
        );
    }
}
