<?php
/**
 * Basket Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use App\User;
use Illuminate\Database\Eloquent\Model;
use Mtc\Shop\Abstracts\PriceModifier;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Events\BasketTotal;
use Mtc\Shop\Events\ItemAddedToBasket;
use Mtc\Shop\PriceMethods\Price;

/**
 * Base Basket model.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Basket extends Model implements BasketContract
{
    /**
     * label that represents the shipping method type for surcharges
     */
    const SHIPPING_METHOD_SURCHARGE_TYPE = 'shipping';

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'array',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'meta',
        'email',
        'phone',
    ];

    /**
     * Set the accessors to append to model arrays.
     *
     * @var array
     */
    protected $appends = [
        'cost_subtotal',
        'cost_tax',
        'cost_total',
    ];

    /**
     * Capture the boot event to save the basket_id within
     * the session if basket created.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::created(
            function ($basket) {
                session(['basket_id' => $basket->id]);
            }
        );
    }

    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(BasketAddress::class);
    }

    /**
     * Get the billing address associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function billingAddress()
    {
        return $this->hasOne(BasketAddress::class)
            ->where('type', 'billing');
    }

    /**
     * Get the shipping address associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function shippingAddress()
    {
        return $this->hasOne(BasketAddress::class)
            ->where('type', 'shipping');
    }

    /**
     * Get the list of items within the basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(BasketItem::class);
    }

    /**
     * Get the list of discounts applied to the basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discounts()
    {
        return $this->hasMany(BasketDiscount::class);
    }

    /**
     * Get the list of surcharges applied to the basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function surcharges()
    {
        return $this->hasMany(BasketSurcharge::class);
    }

    /**
     * Get the delivery method applied to this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function shippingMethod()
    {
        return $this->hasOne(BasketSurcharge::class)
            ->where('type', self::SHIPPING_METHOD_SURCHARGE_TYPE);
    }

    /**
     * Get the user this belongs to
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Add an item to the basket. If basket doesn't exist,
     * it will create it.
     *
     * @param Variant $variant  Variant model
     * @param integer $quantity Number of items to add to basket
     *
     * @return bool
     */
    public function addItem(Variant $variant, $quantity = 1)
    {
        // Get any existing item (if exists)
        $item = $this->items()
            ->where('variant_id', $variant->id)
            ->first() ?? new BasketItem;

        $item->variant_id = $variant->id;
        $item->quantity += $quantity;

        // If we don't have a basket yet, spool it up.
        if (!$this->exists) {
            $this->save();
        }


        event(new ItemAddedToBasket($this, $variant, $quantity));

        return $this->items()->save($item);
    }

    /**
     * Get a value for a particular key in an address, first checking any
     * old data before showing data from the model.
     *
     * @param string $key  Key to check
     * @param string $type Address type to check (billing/shipping)
     *
     * @return string|null Value of key
     */
    public function getAddressValue($key, $type = 'billing')
    {
        if (old("addresses.{$type}.{$key}")) {
            return old("addresses.{$type}.{$key}");
        }

        $address = $this->addresses
            ->where('type', $type)
            ->first();

        if ($address) {
            return $address[$key];
        }

        return null;
    }

    /**
     * Get price modifiers for the final price
     *
     * @return \Illuminate\Support\Collection
     */
    public function getModifiersAttribute()
    {
        return collect(event(new BasketTotal($this)))
            ->filter(
                function ($item) {
                    return is_subclass_of($item, PriceModifier::class);
                }
            );
    }

    /**
     * Get the shipping zone for the selected basket
     *
     * @return int
     */
    public function getShippingZone() : int
    {
        // Return zone that matches country set
        $zone = ShippingZone::whereHas('countries', function ($query) {
            return $query->where('code', $this->shippingAddress->country);
        })
            ->pluck('id')
            ->first();

        // TODO: Region/state checks

        // Return zone with fallback to 0 for no zone
        return $zone ?? 0;
    }

    /**
     * Get the subtotal cost for all items in the basket, without tax.
     *
     * @return int
     */
    public function getCostSubtotalWithoutTaxAttribute() : int
    {
        return $this->items->map(function ($item) {
            return $item->variant->prices->multiple($item->quantity)->price_excluding_tax * $item->quantity;
        })->sum();
    }

    /**
     * Get the subtotal cost for all items in the basket, without tax.
     *
     * @return int
     */
    public function getCostSubtotalAttribute($with_vat = true) : int
    {
        return $this->items->map(function ($item) {
            return $item->getProcessedItem()->price_total;
        })->sum();
    }

    /**
     * Get the total tax added to the items within the basket.
     *
     * @return int
     */
    public function getCostTaxAttribute() : int
    {
        // Tax on items
        $tax_on_basket_items = $this->items->map(function ($item) {
            return $item->quantity * $item->variant
                    ->prices
                    ->multiple($item->quantity)
                    ->tax_on_price;
        })->sum();

        // Tax on surcharges

        $tax_on_surcharges =  $this->surcharges
            ->map(function ($surcharge) {
                return (new Price($surcharge->value))->tax_on_price;
            })->sum();

        // Tax on discounts
        $tax_on_discounts =  $this->discounts
            ->map(function ($discount) {
                if (!empty($discount->amount)) {
                    return $discount->amount->tax_on_price;
                }
            })->sum();

        if ($tax_on_basket_items + $tax_on_surcharges - $tax_on_discounts < 0) {
            return 0;
        }

        return $tax_on_basket_items + $tax_on_surcharges - $tax_on_discounts;
    }

    /**
     * Load all surcharges for the basket
     * During loading process make sure all surcharges have the price set up
     */
    public function loadSurcharges()
    {
        $this->surcharges->map(function ($surcharge) {
            $surcharge->amount = new Price($surcharge->value);
            return $surcharge;
        });
    }


    /**
     * Calculate the total, including any price modifiers.
     *
     * @return int
     */
    public function getCostTotalAttribute() : int
    {
        $total = $this->cost_subtotal;

        // Add surcharges to total
        $total += $this->surcharges
            ->map(function ($surcharge) {
                return (new Price($surcharge->value))->price;
            })->sum();

        // Remove discounts from totals
        $total -= $this->discounts
            ->map(function ($discount) {
                if (!empty($discount->amount)) {
                    return $discount->amount->price;
                }
            })->sum();


        // Add tax to total if not inclusive
        if (false === \Config::get('tax.display_product_tax', true)) {
            $total += $this->cost_tax;
        }

        // Ensure the total is always > 0
        return $total > 0 ? $total : 0;
    }
}
