<?php
/**
 * Price Per Variant Price Method
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Mtc\Shop\Contracts\PriceMethod;
use Mtc\Shop\PriceMethods\Models\PricePerVariant;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Create a 'Per Variant' price method.
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerVariant implements PriceMethod
{
    /**
     * Store the product model
     *
     * @var null|Product
     */
    protected $product = null;

    /**
     * Store the variant model
     *
     * @var null|Variant
     */
    protected $variant = null;

    /**
     * Setup the price method with the product and variant models.
     *
     * @param Mtc\Shop\Product      $product Product this is for
     * @param Mtc\Shop\Variant|null $variant Variant model
     *
     * @return void
     */
    public function __construct(Product $product, Variant $variant = null)
    {
        $this->product = $product;
        $this->variant = $variant;
    }

    /**
     * Get the current price for a single item
     *
     * @param integer $quantity (default: 1)
     *
     * @return Price
     */
    public function single()
    {
        return $this->multiple(1);
    }

    /**
     * Get the price for an item if the user requests the provided quantity
     *
     * @param integer $quantity (default: 1)
     *
     * @return Price
     */
    public function multiple(int $quantity = 1): Price
    {
        $price = null;
        // Bulk pricing is built in by default.
        if ($this->variant) {
            $price = PricePerVariant::where('variant_id', $this->variant->id)
                ->where('quantity', '<=', $quantity)
                ->orderBy('quantity', 'asc')
                ->first();
        } else {
            // if we've asked for a price at product level and therefore not defined a variant,
            // we can still return a price if there are any defined, by  getting all the per variant prices
            // for this product and returning the lowest one (so prices at product level are 'from £x'
            $product = $this->product;
            $price = PricePerVariant::whereHas('variant', function ($query) use ($product) {
                return $query->where('product_id', $product->id);
            })
                ->where('quantity', '<=', $quantity)
                // order by price because we don't know which one they're going for and may as well return an low price
                ->orderBy('price', 'asc')
                ->first();
        }

        if ($price == null) {
            // no prices have been defined
            return new Price(0);
        }

        return new Price($price->price ?: 0);
    }

    /**
     * Get the min/max price for an product.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float[] A 'min' and 'max' returned in an array.
     */
    public function range(int $quantity = 1)
    {
        if ($this->variant) {
            $prices = PricePerVariant::where('variant_id', $this->variant->id)
                ->orderBy('price', 'asc')
                ->get();
        } else {
            $prices = PricePerVariant::whereIn('variant_id', $this->product->variants->pluck('id'))
                ->orderBy('price', 'asc')
                ->get();
        }

        return [
            'min' => new Price($prices->first()->price),
            'max' => new Price($prices->last()->price),
        ];
    }
}
