<?php
/**
 * Payment Gateway Abstract Class
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Abstracts
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Abstracts;

/**
 * Core Payment Gateway Class
 *
 * Extended by individual payment gateways to handle payments, based off of
 * WC_Payment_Gateway.
 *
 * @category Mtc\Shop\Abstracts
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
abstract class PaymentGateway
{
    /**
     * Set if the place order button should be renamed on selection.
     *
     * @var string
     */
    public $order_button_text;

    /**
     * True/false based on whether the method is enabled.
     *
     * @var bool
     */
    public $enabled = true;

    /**
     * Payment method title for the frontend.
     *
     * @var string
     */
    public $title;

    /**
     * Payment method description for the frontend.
     *
     * @var string
     */
    public $description;

    /**
     * Chosen payment method id.
     *
     * @var bool
     */
    public $chosen;

    /**
     * Gateway title.
     *
     * @var string
     */
    public $method_title = '';

    /**
     * Gateway description.
     *
     * @var string
     */
    public $method_description = '';

    /**
     * True if the gateway shows fields on the checkout.
     *
     * @var bool
     */
    public $has_fields;

    /**
     * Countries this gateway is allowed for.
     *
     * @var array
     */
    public $countries;

    /**
     * Available for all counties or specific.
     *
     * @var string
     */
    public $availability;

    /**
     * Icon for the gateway.
     *
     * @var string
     */
    public $icon;

    /**
     * Supported features such as 'products', 'refunds'.
     *
     * @var array
     */
    public $supports = ['products'];

    /**
     * Maximum transaction amount, zero does not define a maximum.
     *
     * @var int
     */
    public $max_amount = 0;

    /**
     * Optional URL to view a transaction.
     *
     * @var string
     */
    public $view_transaction_url = '';

    /**
     * Optional label to show for "new payment method" in the payment
     * method/token selection radio selection.
     *
     * @var string
     */
    public $new_method_label = '';

    /**
     * Contains a users saved tokens for this gateway.
     *
     * @var array
     */
    protected $tokens = [];

    /**
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {

    }

    /**
     * Get the return url (thank you page).
     *
     * @param Order $order Incoming Order
     *
     * @return string URL to redirect to
     */
    public function getReturnUrl($order = null)
    {
        // @todo
    }

    /**
     * Get a link to the transaction on the 3rd party gateway size (if applicable).
     *
     * @param Order $order the order object.
     *
     * @return string transaction URL, or empty string.
     */
    public static function getTransactionUrl($order)
    {
        // @todo
    }

    /**
     * Get the order total in checkout and pay_for_order.
     *
     * @return float
     */
    protected function getOrderTotal()
    {
        // @todo
    }

    /**
     * Check if the gateway is available for use.
     *
     * @return bool
     */
    public function isAvailable()
    {
        // @todo
    }

    /**
     * Set as current gateway.
     *
     * Set this as the current gateway.
     */
    public function setCurrent()
    {
        $this->chosen = true;
    }

    /**
     * Process Payment.
     *
     * Process the payment. Override this in your gateway. When implemented, this should.
     * return the success and redirect in an array. e.g:
     *
     *        return array(
     *            'result'   => 'success',
     *            'redirect' => $this->get_return_url( $order )
     *        );
     *
     * @param int $order_id
     *
     * @return array
     */
    public function processPayment($order_id)
    {
        return array();
    }

    /**
     * Process refund.
     *
     * If the gateway declares 'refunds' support, this will allow it to refund.
     * a passed in amount.
     *
     * @param int    $order_id
     * @param float  $amount
     * @param string $reason
     *
     * @return boolean True or false based on success
     */
    public function processRefund($order_id, $amount = null, $reason = '')
    {
        return false;
    }

    /**
     * Validate frontend fields.
     *
     * Validate payment fields on the frontend.
     *
     * @return bool
     */
    public function validateFields()
    {
        return true;
    }
}
