<?php
/**
 * Discount Servic
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Contracts
 * @package  Mtc\Shop
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Event;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Http\Controllers\CheckoutController;

/**
 * This interface defines the necessary methods for building a Discount
 *
 * @category Mtc\Shop\Contracts
 * @package  Mtc\Shop
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Discount
{
    /**
     * Get the registered discount providers/
     * This method pulls all available discount providers so discounts can be added during checkout process
     *
     * @return string[] list of class names that can provide a discount
     */
    public static function getDiscountProviders()
    {
        return array_filter(Event::fire('Discount::getList'));
    }

    /**
     * Load discounts onto the basket instance
     * This function fetches all Discounts registered on system and applies ones that are available
     *
     * @param Request $request
     * @param BasketContract $basket
     */
    public static function loadDiscounts(Request $request, BasketContract $basket)
    {
        $enabled_discount_providers = self::getDiscountProviders();

        // If no discounts exist stop the process
        if (empty($enabled_discount_providers)) {
            return;
        }

        // Initialize discount forms collection & action for removing a discount from basket
        $basket->discount_forms = collect([]);
        $basket->remove_discount_action = action('\\' . CheckoutController::class . '@removeDiscount');

        // Set up a pipeline to move through all discounts
        $pipeline = app('Illuminate\Pipeline\Pipeline');

        /*
         * Run the pipeline
         * This sends the Basket through a pipeline of $enabled_discount_providers
         * This way all discount providers receive the basket and
         * can add on information about discounts applied & discounts available
         */
        $pipeline->send($basket)
            ->through($enabled_discount_providers)
            ->via('loadDiscount')
            ->then(function ($basket) {
                return $basket;
            });

    }
}
