<?php
/**
 * Browse Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Models\Seo\BrowseContent;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Mtc\Shop\Contracts\ProductRepositoryContract;
use Mtc\Shop\Product;
use Illuminate\Support\Collection;
use Mtc\Shop\Repositories\ProductRepository;

/**
 * Actions revolving around displaying the browse page
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class BrowseController extends Controller
{
    /**
     * @var bool $return_redirect Whether the response should be a redirect
     */
    public $return_redirect = false;

    /**
     * The options available by default for sorting items.
     *
     * @var array
     */
    public $sort_options = [
        [
            'key' => 'most-popular',
            'label' => 'Most Popular'
        ],
        [
            'key' => 'new-additions',
            'label' => 'New Additions'
        ],
        [
            'key' => 'price-asc',
            'label' => 'Price (Low-High)'
        ],
        [
            'key' => 'price-desc',
            'label' => 'Price (High-Low)'
        ],
    ];

    /**
     * BrowseController constructor.
     * Initialize the controller
     *
     * @param string $custom_base_url base url for shop section
     * @param Collection $base_url_taxonomies taxonomies bound to this shop section
     */
    public function __construct($custom_base_url = null, $base_url_taxonomies = null)
    {
        parent::__construct();
        // Set show breadcrumbs as false as we need to make these dynamic
        $this->show_breadcrubs = false;
        $this->setBaseUrl($custom_base_url, $base_url_taxonomies);
    }

    /**
     * Make sure we set the correct base url for this controller
     * This allows us to override the route default path and add other base url
     * alongside with taxonomies that are bound with it.
     * This setup allows us to create custom urls that lead to shop and don't start with /browse
     *
     * @param string $custom_base_url base url for shop section
     * @param Collection $base_url_taxonomies taxonomies bound to this shop section
     */
    protected function setBaseUrl($custom_base_url = null, $base_url_taxonomies = null)
    {
        $this->base_url = $custom_base_url ?? route('shop.browse') . '/';
        $this->base_url_taxonomies = $base_url_taxonomies ?? collect([]);
    }

    /**
     * Display the main page. Even though it's handled by Vue, we need to first
     * send the data syncronously otherwise Google and other bots are unable
     * to read the data.
     *
     * @param Request $request Incoming request (used to
     *                                               check if JSON is wanted)
     * @param ProductRepositoryContract $repository Repository to run queries
     *                                               against
     * @param string $query Search query string
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View|array JSON if requested or regular view
     */
    public function index(Request $request, ProductRepositoryContract $repository, $query = '')
    {
        // Add in default breadcrumb for shop
        $this->breadcrumbs->addCrumb(trans('core::text.shop'), action('\\' . BrowseController::class . '@index'));

        if (false == $request->wantsJson()) {
            $request->merge($this->convertQuery($query));
        }

        $json = $this->getBrowseData($request, $repository, $query);

        if (false === $request->wantsJson()) {
            $data = $this->convertQuery($query, true);
            $data['json'] = $json;

            return view('shop::public.browse')->with($data);
        }

        return $json;
    }

    /**
     * Return the products, options, current URL, and sort options for this page
     *
     * @param Request $request Incoming Request
     * @param ProductRepositoryContract $repository Repository to run queries
     * @param string $query Search query string
     *
     * @return array Compacted list of items
     */
    protected function getBrowseData(
        Request $request,
        ProductRepositoryContract $repository,
        $query = ''
    ) {
        $products = $repository->filter($request);
        $selected = $this->getSelected($request->input('selected', []));

        // Store selected taxonomies in session to keep correct track of how user lands on products
        $request->session()->put('user_journey.browse.taxonomies', $selected);

        $selected->each(function ($selection) use ($request) {
            $this->breadcrumbs->addCrumb($selection['title'], $this->getUrl($request, [$selection]));
        });

        $url = rtrim($this->getUrl($request), '/');
        $sort_options = $this->sort_options;
        $options = $this->getOptions($products, $selected);
        $seo_content = BrowseContent::getContentForSelection($request);
        $return_redirect = $this->return_redirect;
        $crumbs = $this->breadcrumbs->getBreadcrumbs();

        return compact(
            'products',
            'options',
            'url',
            'sort_options',
            'seo_content',
            'return_redirect',
            'crumbs'
        );

    }

    /**
     * Get the taxonomies to show on the filter
     *
     * @param  ProductRepository $products
     * @param  Collection $selected
     * @return Collection
     */
    protected function getOptions($products, Collection $selected): Collection
    {
        // Get the root taxonomies
        $roots = Taxonomy::roots()->with('children')
            ->get()
            ->map(function ($root) use ($selected) {
                $root->selected = $selected->filter(function ($item) use ($root) {
                    return $item->isSelfOrDescendantOf($root);
                });

                $last = $root->selected->last();

                if ($last !== null && $last->children->isEmpty() === false) {
                    $root = $last;
                } elseif ($last !== null) {
                    $root = $last->parent;
                }

                $root->children = $root->children->map(function ($item) use ($selected) {
                    $item->selected = !$selected->where('id', $item->id)->isEmpty();

                    $item->product_count = $item->getDescendantsAndSelf(['id'])->map(function ($item) {
                        return $item->nodes()
                            ->whereNodeableType(Product::class)
                            ->public()
                            ->count();
                    })->sum();

                    return $item;
                });

                // Seems like ->reject() not working here?
                // Maybe because we're overriding the relationship?
                // This seems to fix annoyingly.
                $root = $root->toArray();
                $root['children'] = array_filter($root['children'], function ($item) {
                    return $item['product_count'] > 0;
                });

                return $root;
            });

        return $roots;
    }

    /**
     * Take the incoming URL and parse this into the selected variables, sort,
     * and pagination.
     *
     * @param string $query Incoming query string
     * @param bool $with_root_taxonomies whether to include root taxonomies into selected params
     * @return array
     */
    protected function convertQuery($query, $with_root_taxonomies = true): array
    {
        // Set any defaults
        $data = [
            'query' => '',
            'selected' => collect([]),
            'sort_by' => '',
            'page' => 1,
            'full' => true,
        ];

        // Get the sort/page information
        preg_match_all("/\/?(sort|page)-([-_\w]+)?/", $query, $matches);
        collect($matches)->transpose()->each(
            function ($item) use (&$data) {
                $item[1] = ($item[1] == 'sort' ? 'sort_by' : $item[1]);
                $data[$item[1]] = $item[2];
            }
        );

        // + will be used to separate items within the same taxonomy
        $query = str_replace('+', '/', $query);
        // / will be used to seperate different terms
        $slugs = collect(explode('/', $query))
            ->reject(
                function ($item) {
                    // Remove sort/pagination (just in case)
                    return empty($item) || preg_match('/(?:sort|page)-.*/', $item) > 0;
                }
            );

        // If we have potential nodes, get information about them.
        if ($slugs->isEmpty() === false) {
            $data['selected'] = Node::select(['nodeable_id as id', 'title'])
                ->whereIn('slug', $slugs)
                ->get();
        }

        if (!empty($this->base_url_taxonomies) && $with_root_taxonomies) {
            $this->base_url_taxonomies->each(function ($taxonomy) use ($data) {
                $data['selected']->push($taxonomy);
            });
        }

        return $data;
    }

    /**
     * Generate the url based on controller / request given data
     *
     * @param Request $request incoming request
     * @param array $selected known selections
     * @param bool $full_url whether to show full url or is there a different base url
     * @return string url
     */
    public function getUrl(Request $request, $selected = [], $full_url = true)
    {
        // Get a list of all IDs selected.
        $selected = empty($selected) ? $request->input('selected', []) : $selected;

        // Determine sort of page options
        $additional = [];
        if ($request->input('sort_by')) {
            $additional[] = 'sort-' . $request->input('sort_by');
        }
        if ($request->input('page') && $request->input('page') > 1) {
            $additional[] = 'page-' . $request->input('page');
        }
        $root_stub = $full_url ? $this->base_url : '';

        return self::generateUrl($selected, $root_stub, $additional, $this->base_url_taxonomies);
    }


    /**
     * Get the selected taxonomies
     *
     * @param  mixed $selected
     * @return Collection
     */
    private function getSelected($selected): Collection
    {
        // As this comes in an array with 'id' and 'title', only get the 'id'.
        $selected = (collect($selected))->pluck('id');

        // If we have some taxonomies coming from base url, add them in
        if (!empty($this->base_url_taxonomies)) {
            $selected = $selected->merge($this->base_url_taxonomies->pluck('id'));
        }

        return !empty($selected) ? Taxonomy::whereIn('id', $selected)->get() : collect([]);
    }

    /**
     * Handle a Custom URL request.
     * Majority of flow will go through the default route (i.e. "browse").
     * However Some taxonomies will go through custom urls for certain taxonomies.
     * e.g. /summer-jeans
     *
     * This request is handled over 404 event through CustomUrl Model and ends up redirected here.
     * This section must handle not only rendering this page but also ensure related urls are showing correctly
     *
     * @param Request $request Incoming Request
     * @param Node $node Node that matches base url for this request
     * @return \Illuminate\Http\Response|\Illuminate\View\View
     */
    public function handleCustomUrlRequest(Request $request, Node $node)
    {
        /*
         * If this is a JSON request it means some params are changed
         * Here we need to check if user has not de-selected some of the base params required for this url to work.
         * If user has removed a taxonomy that is required for the url we will need to use the default url for browse.
         */
        if ($request->wantsJson()) {
            $selection_ids = collect($request->input('selected'))->pluck('id');
            $missing_taxonomies = $node->taxonomies->whereNotIn('id', $selection_ids)->count();
        }

        // Set the base url only if there are no missing taxonomies
        if (empty($missing_taxonomies)) {
            // we need to change the base url to be the node url
            $this->setBaseUrl($node->url . '/', $node->taxonomies);
        } else {
            /*
             * Since this section is called via 404 handler instead of a proper controller
             * and the default url will go back to BrowseController via direct route
             * we need to trigger a redirect to the page to ensure page functions correctly.
             * If we do not trigger a redirect here it will result in CSRF token failure
             * on the next ajax call after this response will be sent back.
             */
            $this->return_redirect = true;
        }


        // We need to get the info that is on top of the custom url and merge together with node taxonomies
        $additional = $this->convertQuery($request->getPathInfo());
        $all_taxonomies = $node->taxonomies->merge($additional['selected']);

        // generate query param based on selected taxonomies
        $query = $this->getUrl($request, $all_taxonomies, true);

        // return browse page by adding the exclude params
        return $this->index($request, new ProductRepository, $query);
    }


    /**
     * Generate the current URL or a URL containing additional options.
     *
     * @param array $selected Selected Taxonomies
     * @param array $additional Additional params like ordering or page
     * @param string $base_url base url for for the page
     * @param Collection $base_taxonomies taxonomies that are included in base_url
     * @param bool $full_url whether the full url should be returned or only the query part
     * @return string Full URL (including site domain)
     */
    public static function generateUrl($selected = [], $base_url = '', $additional = [], $base_taxonomies = false)
    {
        $taxonomy_ids = collect($selected)->pluck('id');
        $taxonomy_ids = Taxonomy::whereIn('id', $taxonomy_ids)->get();

        // Now find the root notes for each.
        $root_nodes = $taxonomy_ids
            ->map(function ($child) {
                return $child->getRoot();
            })->unique()
            ->sortBy('id');

        // If the base_url needs to be absolute, generate the base slug based on controllers url
        if ($base_url === 'absolute') {
            $base_url = action('\\' . self::class . '@index') . '/';
        }

        // Generate the URL based on selections
        return $base_url . $root_nodes->map(
                function ($root) use (&$taxonomy_ids, $base_taxonomies) {
                    // Lets process every root level node (e.g. Categories, Brands etc.)
                    // We need to map selections to see if there are in this Taxonomy
                    return $taxonomy_ids->map(function ($child, $child_index) use ($root, &$taxonomy_ids, $base_taxonomies) {

                        // Check if this $child is a descendant of the Taxonomy (is a Category / Brand)
                        if ($child->isDescendantOf($root)) {

                            // Lets remove the index of this child from selections as it has been added to url
                            // This is done so this $taxonomy_ids entry is not processed again in other Taxonomies
                            $taxonomy_ids->forget($child_index);

                            // If this Taxonomy is already part of the base url we need to ignore it
                            // So it is not added into url once more
                            if ($base_taxonomies && $base_taxonomies->where('id', $child->id)->count()) {
                                return null;
                            }

                            // We return a slug as it will be used in the url
                            return $child->node->slug;
                        }

                        // If the child is not a descendant of this Taxonomy, return nothing
                        return null;
                    })->reject(function ($child) {
                        // If the taxonomy node has no slug, we cannot use it so lets reject it
                        return is_null($child);
                    })
                        // same level slugs are merged together with +
                        ->implode('+');
                }
            )
                // We need to push in additional params like sort & page
                ->push($additional)
                // Flatten everything to a single level array
                ->flatten()
                // Merge everything to a single string using / as glue
                ->implode('/');
    }
}
