<?php
/**
 * Per Variant Save Listener
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods\Listeners
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods\Listeners;

use Mtc\Shop\Events\PriceUpdated;
use Mtc\Shop\PriceMethods\Models\PricePerVariant;
use Mtc\Shop\Variant;
use Request;

/**
 * When a save has been initiated for a PerVariant price method, update
 * the price in the associated model.
 *
 * @category Mtc\Shop\PriceMethods\Listeners
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerVariantSave
{
    /**
     * Load anything required - Laravel gets mad if this isn't here
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * Retrieve the data from the included form and update the price for the
     * variant at it's various quantity levels.
     *
     * @param Variant $variant Variant we are saving against.
     *
     * @return void
     */
    public function handle(Variant $variant)
    {

        if ($variant->product->price_method != 'Mtc\Shop\PriceMethods\PerVariant') {
            // If we're not using PerVariant pricing, we want to avoid bothering about PerVariant prices,
            return;
        }

        $quantities = Request::input('price_per_variant.quantity', []);
        $prices = Request::input('price_per_variant.price', []);

        if (count($quantities) == 0) {
            $quantities[] = 1;
        }
        if (count($prices) == 0) {
            $prices[] = 0;
        }

        // Store a list of all saved price ids so we can delete any old ones.
        $saved_prices = [];

        collect(compact('quantities', 'prices'))
            ->transpose()
            // Ditch any whitespace
            ->map(
                function ($item) {
                    return array_map('trim', $item);
                }
            )
            // Reject any empty inputs
            ->reject(
                function ($item) {
                    return empty($item[0]) || (empty($item[1])  && !is_numeric($item[1]));
                }
            )
            // Save the price and store in the $saved_prices aray
            ->each(
                function ($item) use ($variant, &$saved_prices) {
                    $price = PricePerVariant::firstOrNew(
                        [
                        'variant_id' => $variant->id,
                        'quantity' => $item[0],
                        ]
                    );

                    $price->price = intval($item[1] * 100);
                    $price->save();

                    $saved_prices[] = $price->id;
                }
            );

        // Delete any non-existing prices for this variant
        PricePerVariant::whereVariantId($variant->id)
            ->whereNotIn('id', $saved_prices)
            ->delete();

        // Let the system know the price has been updated so sort price and
        // other things can be actioned.
        event(new PriceUpdated($variant->product));
    }
}
