<?php
/**
 * Shop Event Service Provider
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Providers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Providers;

use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Mtc\Core\Admin\Menu;
use Mtc\Core\Admin\Tab;
use Mtc\Core\Events\Admin\MenuLoading;
use Mtc\Core\Events\Admin\RegisterCustomGroupModels;
use Mtc\Shop\Events\ItemAddedToBasket;
use Mtc\Shop\Events\OrderPaid;
use Mtc\Shop\Events\OrderStatusUpdated;
use Mtc\Shop\Events\PriceUpdated;
use Mtc\Shop\Http\Controllers\BrowseController;
use Mtc\Shop\Listeners\AddReservedItem;
use Mtc\Shop\Listeners\NotifyOrderStatusEmail;
use Mtc\Shop\Listeners\UpdateSortPrice;
use Mtc\Shop\Listeners\UpdateStockQuantities;
use Mtc\Shop\Listeners\IncrementNumSold;
use Mtc\Shop\Product;
use Mtc\Shop\ShippingMethods\FlatRate;
use Mtc\Shop\Variant;

/**
 * Within this class are all the events which the Shop is listening and
 * reacting to.
 *
 * @category Mtc\Shop\Providers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class EventServiceProvider extends ServiceProvider
{
    /**
     * Register any events for your application.
     *
     * @return void
     */
    public function boot()
    {
        $this->registerAdminMenuEvents();

        Event::listen(
            RegisterCustomGroupModels::class, function ($event) {
            $event->group
                ->registerModel(Product::class)
                ->registerModel(Variant::class);
        }
        );

        Event::listen(
            'core.shop.price_methods', function () {
            return [
                'Mtc\Shop\PriceMethods\PerProduct',
                'Mtc\Shop\PriceMethods\PerVariant',
            ];
        }
        );

        Event::listen(PriceUpdated::class, UpdateSortPrice::class);

        // stock management stuff
        Event::listen(ItemAddedToBasket::class, AddReservedItem::class);
        Event::listen(OrderPaid::class, UpdateStockQuantities::class);
        Event::listen(OrderPaid::class, IncrementNumSold::class);
        Event::listen(OrderStatusUpdated::class, NotifyOrderStatusEmail::class);

        Event::listen('Mtc\Core\Events\Admin\ItemLoading', 'Mtc\Shop\PriceMethods\Listeners\PerProductForm');
        Event::listen('Mtc\Core\Events\Admin\ItemLoading', 'Mtc\Shop\PriceMethods\Listeners\PerVariantForm');

        Event::listen('eloquent.saved: ' . Variant::class, 'Mtc\Shop\PriceMethods\Listeners\PerVariantSave');
        Event::listen('eloquent.deleted: ' . Variant::class, 'Mtc\Shop\PriceMethods\Listeners\PerVariantDelete');
        Event::listen('eloquent.deleted: ' . Variant::class, 'Mtc\Shop\Listeners\CreateDefaultVariant');
        Event::listen('eloquent.created: ' . Variant::class, 'Mtc\Shop\Listeners\DeleteDefaultVariant');

        Event::listen('eloquent.saved: ' . Product::class, 'Mtc\Shop\PriceMethods\Listeners\PerProductSave');
        Event::listen('eloquent.deleted: ' . Product::class, 'Mtc\Shop\PriceMethods\Listeners\PerProductDelete');
        Event::listen('eloquent.created: ' . Product::class, 'Mtc\Shop\Listeners\CreateDefaultVariant');

        // Listen for custom url controllers, report BrowseController as it can have custom urls
        Event::listen(
            'core.seo.getCustomUrlControllers',
            function () {
                return [
                    BrowseController::class
                ];
            }
        );

        // Listen for Menu builder checking menu types
        Event::listen(
            'Mtc\Menus\Events\Admin\GetMenuSections',
            'Mtc\Shop\Listeners\BrowseMenuSection'
        );

        Event::listen(
            'tabs.loading: ' . Product::class, function ($model) {
            // if no model id only show tabs that relate to product without ID
            if (empty($model->id)) {
                return [
                    new Tab(
                        'products.create',
                        'Details',
                        route('shop.admin.products.create', $model)
                    )
                ];
            }
            return [
                new Tab(
                    'products.edit',
                    'Details',
                    route('shop.admin.products.edit', $model)
                ),
                new Tab(
                    'variants.index',
                    'Variants & Stock',
                    route('shop.admin.variants.index', $model)
                ),
                new Tab(
                    'media.index',
                    'Media',
                    route('shop.admin.media.index', $model)
                )
            ];
        }
        );

        Event::listen(
            'shop.shipping_methods', function () {
            return new FlatRate;
        }
        );
    }

    /**
     * Register menu events for admin area
     */
    private function registerAdminMenuEvents()
    {

        Event::listen(
            MenuLoading::class, function () {
            Menu::addSection(
                'Shop',
                'shop'
            );

            Menu::addMenu(
                'shop',
                'Products',
                '',
                'products',
                'glyphicon glyphicon-barcode'
            );
            Menu::addSubMenu(
                'products',
                'Manage Products',
                '',
                route('shop.admin.products.index'),
                'manage_products'
            );

            if (config('stock.stock_enabled')) {
                // add menu item for stock management if the functionality has been enabled
                Menu::AddSubMenu('products',
                    'Stock Settings',
                    '',
                    route('shop.admin.stocksettings.edit'),
                    'stock_settings'
                );
            }

            Menu::addMenu(
                'shop',
                'Orders',
                '',
                'orders',
                'glyphicon glyphicon-briefcase'
            );
            Menu::addSubMenu(
                'orders',
                'Manage Orders',
                '',
                route('shop.admin.orders.index'),
                'manage_orders'
            );
            if (config('order.picking_list')) {
                // add menu item for order picking lists if the functionality has been enabled
                Menu::addSubMenu(
                    'orders',
                    'Picking List',
                    '',
                    route('shop.admin.orders.picking'),
                    'manage_orders'
                );
            }

            Menu::addMenu(
                'shop',
                'Shipping',
                '',
                'shipping',
                'glyphicon glyphicon-plane'
            );
            Menu::addSubMenu(
                'shipping',
                'Zones',
                '',
                route('shop.admin.shipping.zones.index'),
                'manage_shipping_zones'
            );
            Menu::addSubMenu(
                'shipping',
                'Flat Rates',
                '',
                route('shop.admin.shipping.flat-rate.index'),
                'manage_shipping_rates'
            );

            Menu::addMenu(
                'shop',
                'Discounts',
                '',
                'discounts',
                'glyphicon glyphicon-money'
            );
        }
        );
    }
}
