<?php
/**
 * Product Variant Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Session;
use Mtc\Core\Nodeable;
use Mtc\Shop\Stock\Reserved as StockReserved;
use Mtc\Shop\Stock\Settings as StockSettings;

/**
 * Store information about a variant/size of a product.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Variant extends Model
{
    use Nodeable;
    use Pricing;
    /**
     * The attributes filled out by default when a new model is created.
     *
     * @var array
     */
    protected $attributes = [
        'stock_enabled' => false,
        'stock_quantity' => 0
    ];
    /**
     * The attributes to load with this object when instatiating
     *
     * @var array
     */
    protected $with = ['node'];
    /**
     * The attributes that are mass assignable
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'stock_enabled',
        'stock_quantity',
        'shipping_weight',
        'shipping_dimensions_length',
        'shipping_dimensions_width',
        'shipping_dimensions_height'
    ];
    /**
     * Update parent model timestamps.
     *
     * @var array
     */
    protected $touches = ['product'];
    /**
     * The attributes that should be loaded when the model is loaded.
     *
     * @var array
     */
    protected $appends = ['price'];
    /**
     * Get the parent basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }
    /**
     * Get the Sku code attached to this variant
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphOne
     */
    public function sku()
    {
        return $this->morphOne(Sku::class, 'owner');
    }
    /**
     * get any variants that have been reserved
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function stockReserved()
    {
        return $this->hasMany(StockReserved::class);
    }
    /**
     * Return an admin panel URL for the variant.
     *
     * @param boolean $absolute Include the Site's URL.
     *
     * @return string
     */
    public function getAdminUrl($absolute = true)
    {
        return route('shop.admin.variants.edit',
            [
                'product' => $this->product_id,
                'variant' => $this->id
            ],
            $absolute
        );
    }
    /**
     * Override the value in stock_quantity to take into account your stock settings and reserves
     *
     * @param int $value the real value of the attribute in our database
     *
     * @return int
     */
    public function getStockQuantityAttribute(int $value)
    {
        return $this->getAdjustedStockQuantity($value);
    }
    /**
     * function isInStock
     *
     * A variant is considered to be in stock if its adjusted stock quantity is greater than zero
     * Or if stock tracking is disabled globally
     *
     * @return bool
     */
    public function isInStock()
    {
        return $this->stock_quantity > 0;
    }
    /**
     * return the stock quantity but take into account stock management config and reserved items
     *
     * @param int $stock_quantity the actual database value of stock quantity passed to us by its accessor
     *
     * @return int
     */
    public function getAdjustedStockQuantity(int $stock_quantity)
    {
        $adjusted_stock_quantity = 0;
        if (!config('stock.stock_enabled') || false == ($this->product->stock_enabled || $this->stock_enabled)) {
            // stock is disabled globally, OR stock is disabled at product level and variant level
            return PHP_INT_MAX; // always in stock
        }
        // stock is enabled globally, and stock is either enabled at product level, or for this individual variant
        $reserved_variants_count = 0;
        $reserved_variants = $this->stockReserved()
            ->current()
            ->where('basket_id', '!=', Session::get('basket_id'))
            ->get();
        foreach ($reserved_variants as $reserved_variant) {
            $reserved_variants_count += $reserved_variant->quantity;
        }
        $adjusted_stock_quantity = $stock_quantity
            - $reserved_variants_count
            - (StockSettings::first()->stock_reserve || 0);
        return $adjusted_stock_quantity;
    }
}