<?php

namespace Mtc\Shop\Http\Controllers\Admin;

use Mtc\Core\Http\Controllers\Controller;
use Event;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Order\OrderNote;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, OrderContract $order)
    {
        $query = $order->query()
            ->whereIsPaid(1)
            ->buildSearch($request)
            ->orderBy('id', 'desc');

        // Bring in search
        Event::listen(
            'core.builder.shop.admin.orders.head', function() {
                return 'shop::admin.order.index_head';
            }
        );

        return (
            new Builder('shop.admin.orders', $query))
            ->columns(
                [
                    'id' => trans('fields.id'),
                    'name' => trans('shop::basket.personal.name'),
                    'email' => trans('shop::basket.personal.email'),
                    'status.title' => 'Status',
                    'date' => 'Date',
                ]
            )
            ->data(
                [
                    'name' => function($order) {
                        $first = $order->addresses->first()->first_name;
                        $last = $order->addresses->first()->last_name;
                        return "{$first} {$last}";
                    },
                    'date' => function($order) {
                        return $order->created_at;
                    },
                    'status.title' => function($order) {
                        return $order->status->title;
                    }
                ]
            )
            ->view();
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param OrderContract $order Order Model
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(OrderContract $order)
    {
        $title = "Order {$order->id}";
        $subtitle = $order->created_at->toDayDateTimeString();

        $addresses = $order->addresses->keyBy('type')
            ->map(
                function($address) {
                    // Generate an address with each filled in var as new line
                    $address['full'] = collect($address)
                    ->only(
                        [
                        'address1',
                        'address2',
                        'city',
                        'state',
                        'postcode',
                        'country'
                        ]
                    )
                    ->reject(
                        function($item) {
                            return empty(trim($item));
                        }
                    )
                    ->implode(PHP_EOL);

                    return $address;
                }
            );

        return view('shop::admin.order.edit')
            ->with(compact('order', 'title', 'subtitle', 'addresses'));

        return (new ItemBuilder('shop.admin.orders', $order, null))
            ->view('shop::admin.order.edit');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param int                      $id      Order ID
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param int                      $id      Order ID
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Add a note via the admin area for this order.
     *
     * @param OrderContract $order   Order
     * @param Request       $request Request with 'note' param.
     * @param OrderNote     $note    Blank OrderNote to populate
     *
     * @return \Illuminate\Http\Response Redirect to order page
     */
    public function addNote(OrderContract $order, Request $request, OrderNote $note)
    {
        $this->validate($request, [
            'note' => 'required'
        ]);

        $note->user_id = $request->user()->id;
        $note->message = $request->input('note');

        $order->notes()->save($note);

        return redirect(route('shop.admin.orders.edit', $order) . '#orderNotes');
    }
}
