<?php
/**
 * Product Controller (Admin)
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers\Admin;

use Event;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Node;
use Mtc\Shop\Http\Requests\Admin\StoreProduct;
use Mtc\Shop\Product;

/**
 * Displays the product routes as a resource
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        // Bring in search
        Event::listen(
            'core.builder.shop.admin.products.head', function() {
                return 'shop::admin.product.search';
            }
        );

        return (new Builder('shop.admin.products', Product::orderBy('id', 'desc')
            ->buildSearch($request)))
            ->columns(
                [
                    'id' => trans('fields.id'),
                    'node.title' => trans('fields.title'),
                    'node.status' => trans('fields.status'),
                    'stock_enabled' => trans('fields.stock_enabled'),
                    'updated_at' => trans('fields.updated_at')
                ]
            )
            ->data(
                [
                    'node.title' => function($item) {
                        return $item->node ? $item->node->title : '';
                    },
                    'stock_enabled' => function($item) {
                        return $item->stock_enabled ? 'Yes' : 'No';
                    },
                    'node.status' => function($item) {
                        return $item->node ? ucfirst($item->node->status) : '';
                    }
                ]
            )
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return (new ItemBuilder('shop.admin.products', new Product, new Node))
            ->view('shop::admin.product.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreProduct $request Validated Request
     * @param Product      $product Empty product model
     * @param Node         $node    Empty node model
     *
     * @return \Illuminate\Http\Response Redirect to product home
     */
    public function store(StoreProduct $request, Product $product, Node $node)
    {
        $product->fill($request->input('item', []))->save();
        $node->fill($request->input('node', []));

        $product->node()->save($node);

        return redirect(route('shop.admin.products.index'));
    }

    /**
     * Display the specified resource with it's parent Node
     *
     * @param int $id Product ID
     *
     * @return \Illuminate\Http\Response JSON response
     */
    public function show($id)
    {
        return Product::with('node')->findOrFail($id);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Product $product Product Model
     *
     * @return \Illuminate\View\View
     */
    public function edit(Product $product)
    {
        return (new ItemBuilder('shop.admin.products', $product, $product->node))
            ->view('shop::admin.product.form');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreProduct $request Validated Request
     * @param Product      $product Existing Product model
     *
     * @return \Illuminate\Http\Response Redirect back to product home
     */
    public function update(StoreProduct $request, Product $product)
    {
        $product->fill($request->input('item', []));
        $product->node->fill($request->input('node', []))->save();
        $product->save();

        return redirect(route('shop.admin.products.index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request Incoming Request
     * @param Product $product Product to destroy
     *
     * @return \Illuminate\Http\Response Either JSON success or redirect back
     */
    public function destroy(Request $request, Product $product)
    {
        if ($product->node) {
            $product->node->delete();
        }

        $product->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "{$product->title} has been deleted.");
        return redirect()->back();
    }
}
