<?php
/**
 * Product Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Cache;
use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Media;
use Mtc\Core\Node;
use Mtc\Core\Nodeable;
use Mtc\Shop\PriceMethods\PerProduct;

/**
 * Base Product Model
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Product extends Model
{
    use Nodeable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'price_method',
        'stock_enabled',
        'stock_status',
    ];

    /**
     * Default attributes for a new Product.
     *
     * @var array
     */
    protected $attributes = [
        'price_method' => PerProduct::class,
        'stock_enabled' => false,
    ];

    /**
     * Retrieve the related variants (sizes)
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function variants()
    {
        return $this->hasMany(Variant::class);
    }

    /**
     * Return price method when $this->price is requested.
     *
     * @return \Mtc\Shop\Contracts\PriceMethod
     */
    public function getPriceAttribute()
    {
        $price_method = $this->price_method;

        if (!class_exists($price_method)) {
            throw new \Exception("Price method not found: {$price_method}");
        }

        return new $price_method($this);
    }

    /**
     * Retrieve any related media.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function media()
    {
        return $this->morphMany(Media::class, 'parent');
    }

    /**
     * Scope for Eloquent queries to search parent node for a term.
     *
     * @param \Illuminate\Eloquent\Query\Builder $query Query Builder
     * @param string                             $term  Search Query
     *
     * @return \Illuminate\Eloquent\Query\Builder
     */
    public function scopeSearch($query, $term)
    {
        if (!empty($term)) {
            $query->whereHas(
                'node', function($query) use ($term) {
                    $query->search($term);
                }
            );
        }
        return $query;
    }

    /**
     * Return a URL for the product, caching the URL for 30 minutes.
     *
     * @param boolean                $absolute Include the Site's URL.
     * @param boolean|\Mtc\Core\Node $node     The associated node if available
     *
     * @return string
     */
    public function getUrl($absolute = true, $node = false)
    {
        // As this may be called ahead of the node being saved, we need to
        // pass through.
        if ($node instanceof Node === false) {
            $node = $this->node;
        }

        return Cache::remember(
            "shop.product:{$this->id}:url",
            30,
            function() use ($absolute, $node) {
                return route('shop.product', [$this, $node], $absolute);
            }
        );
    }

    /**
     * Build a search query for the admin panel.
     *
     * @param  \Illuminate\Database\Eloquent\Builder $query   Model Query
     * @param  \Illuminate\Http\Request              $request Browser Request
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeBuildSearch(
        \Illuminate\Database\Eloquent\Builder $query,
        \Illuminate\Http\Request $request
    ) {
        if ($request->has('title')) {
            $query->whereHas('node', function($query) use ($request) {
                $query->where('title', 'LIKE', '%' . $request->input('title') . '%');
            });
        };

        return $query;
    }
}
