<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateOrdersTable extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        /**
         * Used to store the status and email text to send for an order.
         */
        Schema::create('order_statuses', function (Blueprint $table) {
            $table->increments('id');
            $table->string('title');
            $table->boolean('is_email_sent')
                ->default(true);
            $table->text('email_body')
                ->nullable()
                ->default(null);
            $table->timestamps();
            // We don't want delete of statuses for existing orders.
            $table->softDeletes();
        });

        Schema::create('orders', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_status_id')
                ->unsigned();
            $table->integer('basket_id')
                ->unsigned()
                ->nullable()
                ->default(null);
            $table->boolean('is_paid')
                ->default(false);
            $table->string('reference')
                ->nullable()
                ->default(null)
                ->index();
            $table->string('email', 75)
                ->nullable()
                ->default(null);
            $table->string('phone', 50)
                ->nullable()
                ->default(null);
            $table->text('meta')
                ->nullable()
                ->default(null);
            $table->text('payment')
                ->nullable()
                ->default(null);
            $table->decimal('total', 10,2)
                ->unsigned();
            $table->timestamps();

            $table->foreign('order_status_id')
                ->references('id')
                ->on('order_statuses')
                ->onUpdate('cascade')
                ->onDelete('restrict');

            $table->foreign('basket_id')
                ->references('id')
                ->on('baskets')
                ->onUpdate('cascade')
                ->onDelete('set null');
        });

        Schema::create('order_addresses', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')
                ->unsigned();
            $table->enum('type', ['billing', 'shipping'])
                ->index();

            $table->string('full_name', 200)->index();
            $table->string('first_name', 100);
            $table->string('last_name', 100);
            $table->string('address1', 150);
            $table->string('address2', 150)
                ->nullable()
                ->default(null);
            $table->string('city', 80);
            $table->string('state', 25)
                ->nullable()
                ->default(null);
            $table->string('postcode', 10);
            $table->string('country', 2);

            $table->timestamps();

            $table->unique(['order_id', 'type']);

            $table->foreign('order_id')
                ->references('id')
                ->on('orders')
                ->onUpdate('cascade')
                ->onDelete('cascade');
        });

        Schema::create('order_items', function (Blueprint $table) {
            $table->increments('id');
            $table->string('code')->index();
            $table->integer('order_id')->unsigned();
            $table->integer('product_id')->unsigned()->nullable();
            $table->string('product_title');
            $table->integer('variant_id')->unsigned()->nullable();
            $table->string('variant_title');
            $table->smallInteger('quantity')->unsigned();
            $table->decimal('price', 10,2)
                ->unsigned();
            $table->timestamps();

            $table->foreign('order_id')
                ->references('id')
                ->on('orders')
                ->onUpdate('cascade')
                ->onDelete('cascade');
            $table->foreign('product_id')
                ->references('id')
                ->on('products')
                ->onUpdate('cascade')
                ->onDelete('set null');
            $table->foreign('variant_id')
                ->references('id')
                ->on('variants')
                ->onUpdate('cascade')
                ->onDelete('set null');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('order_items');
        Schema::dropIfExists('order_addresses');
        Schema::dropIfExists('orders');
        Schema::dropIfExists('order_statuses');
    }
}
