<?php
/**
 * Product Variant Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Nodeable;

/**
 * Store information about a variant/size of a product.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Variant extends Model
{
    use Nodeable;

    /**
     * The attributes filled out by default when a new model is created.
     *
     * @var array
     */
    protected $attributes = [
        'stock_enabled' => false
    ];

    /**
     * The attributes that are mass assignable
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'code',
        'stock_enabled',
        'stock_quantity',
        'shipping_weight',
        'shipping_dimensions_length',
        'shipping_dimensions_width',
        'shipping_dimensions_height',
        'price',
        'sale_price',
        'sale_price_valid_from',
        'sale_price_valid_until',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'price' => 'double',
        'sale_price' => 'double',
    ];

    /**
     * Update parent model timestamps.
     *
     * @var array
     */
    protected $touches = ['product'];

    /**
     * Get the parent basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * When price is requested via $variant->price, return the Price Method
     * for this product.
     *
     * @throws Exception If price method if not found
     *
     * @return \Mtc\Shop\Contracts\PriceMethod
     */
    public function getPriceAttribute()
    {
        $price_method = $this->product->price_method;

        if (!class_exists($price_method)) {
            throw new \Exception("Price method not found: {$price_method}");
        }

        return new $price_method($this->product, $this);
    }
}
