<?php
/**
 * Product Media Controller (Admin)
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers\Admin;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Media;
use Mtc\Shop\Http\Requests\StoreImage;
use Mtc\Shop\Product;

/**
 * Actions concerning display and upload of media for a product
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class MediaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Product $product Product Model
     *
     * @return \Illuminate\View\View
     */
    public function index(Product $product)
    {
        $size_collection = collect(config('media.' . Product::class))
            ->reject(function ($entry) {
                // Original upload path is a string, it is not a valid path
                return !is_array($entry);
            })
            ->keyBy(function ($size) {
                return ucfirst(basename($size['path']));
            });

        return (new ItemBuilder('shop.admin.product', $product, $product->node))
            ->tab(
                'shop::admin.media.index', [
                    'media' => $product->media->map(
                        function (Media $media) {
                            $media->setDefaultAttributes();
                            return $media;
                        }
                    ),
                    'sizes' => $size_collection,
                    'can_edit_meta' => Auth::user()->can('manage-media-meta')
                ]
            );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreImage $request Validated Request
     * @param Product $product Product to attach media to
     *
     * @return \Illuminate\Http\Response
     */
    public function store(StoreImage $request, Product $product)
    {
        if ($request->input('action') === 'update_meta') {
            return $this->update($request, $product);
        }
        // Find the upload path for the product image
        $upload_path = config('media.' . Product::class . '.upload_path');

        // Save the image in this location
        $path = Storage::disk('public')->put($upload_path, $request->file('file'));

        // Create media object
        $media = new Media([
            'src' => $path,
            'title' => $request->file('file')->getClientOriginalName(),
            'type' => $request->file('file')->getMimeType(),
        ]);

        // Save this media file for the product
        $product->media()->save($media);

        // Add a unique seo friendly name for the image
        $media->url_name = str_slug($product->node->title) . '-' . $product->node->id . '-' . $media->id
            . '.' . $request->file('file')->getClientOriginalExtension();
        $media->save();

        return $product->media->map(
            function ($media) {
                $media->setDefaultAttributes();
                return $media;
            }
        );
    }

    /**
     * Update information for a Media file
     *
     * @param Request $request incoming request
     * @param Product $product Product to whom media files are assigned to
     */
    public function update(Request $request, Product $product)
    {
        foreach ($request->input('media') as $key => $object) {
            $this->validate($request, [
                "media.{$key}.url_name" => 'required|alpha_dash_file|unique:media,url_name,' . $object['id']
            ], [], [
                "media.{$key}.url_name" => 'Image'
            ]);
        }

        // Update image url name and title
        foreach ($request->input('media', []) as $key => $media_request) {

            // Load the media file
            $media = Media::find($media_request['id']);

            if ($media) {
                // update values
                $media->url_name = $media_request['url_name'];
                $media->title = $media_request['title'];
                $media->save();
            }
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id Media ID
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Product $product, $id)
    {
        $media = Media::findOrFail($id)->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Media file '{$media->title}' has been deleted.");
        return redirect()->back();
    }

    /**
     * Update order of images upon re-order
     *
     * @param Request $request incoming request
     * @param Product $product product whose images are in request
     */
    public function changeOrder(Request $request, Product $product)
    {
        foreach ($request->input('media') as $order => $media) {
            Media::where('id', $media['id'])
                ->where('parent_type', Product::class)
                ->where('parent_id', $product->id)
                ->update([
                    'order' => $order
                ]);
        }
    }
}
