<?php

namespace Mtc\Shop\Order;

use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Model;
use Jenssegers\Agent\Agent;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Events\OrderPaid;

class Order extends Model implements OrderContract
{
    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'array',
        'payment' => 'array',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'order_status_id',
        'basket_id',
        'is_paid',
        'reference',
        'email',
        'phone',
        'meta',
        'payment',
        'total',
    ];

    /**
     * Get the addresses associated with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(OrderAddress::class);
    }

    /**
     * Get the discounts associated with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discounts()
    {
        return $this->hasMany(OrderDiscount::class);
    }

    /**
     * Get the surcharges associated with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function surcharges()
    {
        return $this->hasMany(OrderSurcharge::class);
    }

    /**
     * Create a new order from an existing basket
     *
     * @param BasketContract $basket
     *
     * @return self
     */
    public function createFromBasket(BasketContract $basket)
    {
        $this->fill($basket->toArray());
        $this->basket_id = $basket->id;
        $this->order_status_id = 1;
        $this->total = $basket->cost_total;
        $this->save();

        $basket->addresses->each(
            function ($address) {
                $this->addresses()->create($address->toArray());
            }
        );

        $basket->discounts->each(function ($discount) {
            $discount->discount_reference = class_basename($discount->discount_type) . ': ' . $discount->reference;
            $discount->value = $discount->amount->price;
            $this->discounts()->create($discount->toArray());
        });

        $basket->surcharges->each(function ($surcharge) {
            $surcharge->display_name = ucwords($surcharge->type);
            $this->surcharges()->create($surcharge->toArray());
        });

        $basket->items->each(
            function ($item) {
                $this->items()->create(
                    [
                    'sku' => $item->variant->sku->sku ?? '',
                    'product_id' => $item->variant->product_id,
                    'product_title' => $item->variant->product->node->title,
                    'variant_id' => $item->variant_id,
                    'variant_title' => $item->variant->node->title,
                    'quantity' => $item->quantity,
                    'price' => $item->variant->prices->multiple($item->quantity)->price,
                    ]
                );
            }
        );

        // Store info about user agent used for placing an order
        $user_agent = new Agent();
        $this->info()->create([
            'ip' => request()->server('REMOTE_ADDR'),
            'platform' => $user_agent->platform(),
            'platform_version' => $user_agent->version($user_agent->platform()),
            'browser' => $user_agent->browser(),
            'browser_version' => $user_agent->version($user_agent->browser()),
            'device' => $user_agent->device(),
            'device_version' => $user_agent->version($user_agent->device()),
            'languages' => implode(', ', $user_agent->languages()),
        ]);

        return $this->find($this->id);
    }

    /**
     * Get the user info with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function info()
    {
        return $this->hasOne(OrderInfo::class);
    }

    /**
     * Get the items associated with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Mark the order as paid, and in the process remove
     * existing unpaid orders and basket.
     *
     * @return void
     */
    public function markPaid()
    {
        $this->is_paid = true;
        $this->save();

        $this->whereBasketId($this->basket_id)
            ->where('id', '!=', $this->id)
            ->delete();

        resolve(BasketContract::class)
            ->whereId($this->basket_id)
            ->delete();

        session()->forget('basket_id');

        event(new OrderPaid($this));
    }

    /**
     * Get the notes associated with the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function notes()
    {
        return $this->hasMany(OrderNote::class)->orderBy('id', 'desc');
    }

    /**
     * Run a search on the incoming request.
     *
     * @param Illuminate\Eloquent\Query\Builder $query   Query Builder
     * @param Request                           $request Incoming search request
     *
     * @return Illuminate\Eloquent\Query\Builder
     */
    public function scopeBuildSearch($query, Request $request)
    {
        // Keep in seperate query to keep inclusive for OR queries
        $query->where(
            function ($query) use ($request) {
                if ($request->has('status')) {
                    $query->where('order_status_id', '=', $request->input('status'));
                }

                if ($request->has('id')) {
                    $query->where('id', '=', $request->input('id'));
                }

                if ($request->has('email')) {
                    $query->where('email', 'LIKE', '%' . $request->input('email') . '%');
                }

                if ($request->has('name')) {
                    $name = $request->input('name');
                    $query->whereHas(
                        'addresses', function ($query) use ($name) {
                            $query->where('full_name', 'LIKE', "%{$name}%");
                        }
                    );
                }
            }
        );

        return $query;
    }

    /**
     * Get the associated order status.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function status()
    {
        return $this->belongsTo(OrderStatus::class, 'order_status_id');
    }

    /**
     * Get the subtotal cost for all items in the order
     *
     * @return int
     */
    public function getCostSubtotalAttribute() : int
    {
        return $this->items->sum('price');
    }
}
