<?php

namespace Mtc\Shop\ShippingMethods;

use Illuminate\Support\Collection;
use Mtc\Shop\Abstracts\ShippingMethod;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Http\Controllers\Admin\ShippingFlatRateController;
use Mtc\Shop\ShippingFlatRate;

/**
 * Class FlatRate
 *
 *
 * @package Mtc\Shop\ShippingMethods
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-17
 */
class FlatRate extends ShippingMethod
{
    /**
     * @var string $id Shipping Method identificator
     */
    public $id = 'flat-rate';

    /**
     * @var string $title Shipping Method title
     */
    public $title = 'Flat Rate';

    /**
     * Calculate shipping costs
     *
     * @param BasketContract|null $basket
     * @return array
     */
    public function calculateShipping(BasketContract $basket = null): array
    {
        // No basket given means we can't give any rates
        if (!$basket) {
            return [];
        }

        // get basket total & shipping zone from basket
        $basket_total = $basket->getCostSubtotalAttribute();
        $zone = $basket->getShippingZone();

        /*
         * Retrieve all flat rates that are:
         * - tied to selected zone
         * - have min value less than basket total
         * - have no max value or max value higher than basket total
         *
         * These entries then are sorted by cost, keyed by id and returned as an array
         */
        return ShippingFlatRate::where('min_basket_value', '<=', $basket_total)
            ->where(function ($query) use ($basket_total) {
                $query->whereNull('max_basket_value')
                    ->orWhere('max_basket_value', '>=', $basket_total);
            })
            ->where('shipping_zone_id', $zone)
            ->get()
            ->map(function ($flat_rate) {
                $flat_rate->class = ShippingFlatRate::class;
                return $flat_rate;
            })
            ->sortBy('value')
            ->keyBy('id')
            ->toArray();
    }

    /**
     * Flat rates always can be edited/added
     *
     * @return string
     */
    public function isManageable()
    {
        return true;
    }

    /**
     * Get manage url for a flat rate method management (list / edit view)
     *
     * @param int $zone_id Selected Zone whose rates to display
     * @param int $rate_id Specific rate to edit
     * @return string url to action
     */
    public function manageUrl($zone_id, $rate_id = false): string
    {
        // Find Rate
        $rate = ShippingFlatRate::find($rate_id);

        // If we have rate we can open single rate view
        if ($rate) {
            return route(
                ShippingFlatRateController::$route_name . ".edit",
                [
                    'zone' => $zone_id,
                    'rate' => $rate->id
                ]
            );
        }

        // return index page
        return route(
            ShippingFlatRateController::$route_name . ".index",
            [
                'zone' => $zone_id
            ]
        );
    }

    /**
     * Get a breakdown for Flat rates for a specific zone
     *
     * @param int $shipping_zone_id
     * @return mixed
     */
    public function getCostBreakdown($shipping_zone_id): Collection
    {
        return ShippingFlatRate::where('shipping_zone_id', $shipping_zone_id)
            ->orderBy('min_basket_value', 'asc')
            ->get();
    }
}
