<?php
/**
 * Class SlideShowController
 *
 * @package Mtc\Slideshow\Http\Controllers\Admin
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Slideshow\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Node;
use Mtc\Slideshow\Models\Slider;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Response;
use Illuminate\Routing\Redirector;
use Illuminate\View\View;
use Illuminate\Contracts\Routing\ResponseFactory;
use Symfony\Component\HttpFoundation\Response as SymfonyResponse;

/**
 * Class SlideShowController
 *
 * SlideShow (slider) management.
 * Allows creating new sliders
 *
 * @package Mtc\Slideshow\Http\Controllers\Admin
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-06-20
 */
class SlideShowController extends Controller
{
    /**
     * List all existing sliders.
     *
     * If Multiple sliders is disabled, redirect to main slider page
     *
     * @param Request $request
     * @return RedirectResponse|Response|Redirector|View
     */
    public function index(Request $request)
    {
        // If we are in single slider mode
        if (!config('slideshow.multiple_sliders')) {
            $slider = Slider::orderBy('id', 'asc')
                ->take(1)
                ->get()
                ->first();

            // if there is not a slider, we need to make one
            if (!$slider) {
                $slider = new Slider();
                $slider->save();
                $slider->node()->create([
                    'title' => 'Main Slider',
                    'status' => 'published',
                    'visibility' => 'public'
                ]);
            }
            return redirect(action('\\' . SlideController::class . '@index', [$slider->id]));
        }

        return (new Builder(
            'slider.admin.slide-show',
            Slider::orderBy('id', 'asc')->buildSearch($request)
        ))
            ->columns(
                [
                    'id' => trans('fields.id'),
                    'node.title' => trans('fields.title'),
                    'node.status' => trans('fields.status'),
                    'updated_at' => trans('fields.updated_at')
                ]
            )
            ->data(
                [
                    'node.title' => function ($item) {
                        return $item->node ? $item->node->title : '';
                    },
                    'node.status' => function ($item) {
                        return $item->node ? ucfirst($item->node->status) : '';
                    }
                ]
            )
            ->view();
    }

    /**
     * Create a new slider
     *
     * @return View
     */
    public function create()
    {
        return (new ItemBuilder('slider.admin.slide-show', new Slider, new Node))
            ->view();

    }

    /**
     * Store a new Slider
     *
     * @param Request $request incoming request
     * @param Slider $slider Empty Slider instance
     * @param Node $node Empty Node instance
     * @return RedirectResponse|Redirector
     */
    public function store(Request $request, Slider $slider, Node $node)
    {
        $slider->save();
        $node->fill($request->input('node', []));
        $slider->node()->save($node);

        $request->session()->flash('info', trans('slideshow::messages.created'));
        return redirect(route('slider.admin.slide-show.edit', $slider));
    }

    /**
     * Edit an existing slider
     *
     * @param Slider $slider Slider to edit
     * @return View
     */
    public function edit(Slider $slider)
    {
        return (new ItemBuilder('slider.admin.slide-show', $slider, $slider->node))
            ->view();
    }

    /**
     * Update an existing slider
     *
     * @param Request $request incoming request
     * @param Slider $slider Slider to edit
     * @return RedirectResponse|Redirector
     */
    public function update(Request $request, Slider $slider)
    {
        $slider->node->fill($request->input('node', []));
        $slider->node->save();

        $request->session()->flash('info', trans('slideshow::messages.updated'));
        return redirect(route('slider.admin.slide-show.edit', $slider));
    }

    /**
     * Delete a slider
     * @param Request $request
     * @param Slider $slider
     * @return ResponseFactory|RedirectResponse|SymfonyResponse
     */
    public function destroy(Request $request, Slider $slider)
    {

        $slider->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', trans("slideshow::messages.deleted", [
            'slider_name' => $slider->node->title
        ]));
        return redirect()->back();
    }
}