<?php
/**
 * Class SlideController
 *
 * @package Mtc\Slideshow
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Slideshow\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Media;
use Mtc\Core\Node;
use Mtc\Shop\Http\Requests\StoreImage;
use Mtc\Slideshow\Models\Slide;
use Mtc\Slideshow\Models\Slider;
use Illuminate\Http\RedirectResponse;
use Illuminate\Routing\Redirector;

/**
 * Class SlideController
 *
 * Allows managing slides of a slider
 *
 * @package Mtc\Slideshow
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-06-20
 */
class SlideController extends Controller
{
    /**
     * List all slides of $slider
     *
     * @param Slider $slider slider whose slides to display
     * @return \Illuminate\View\View
     */
    public function index(Slider $slider)
    {
        return (new ItemBuilder('slider.admin.slide-show', $slider, $slider->node))
            ->tab(
                'slideshow::admin.slider.slides', [
                    'media' => $this->getAdminSlides($slider),
                    'sizes' => $this->getSizeCollection(),
                    'can_edit_meta' => Auth::user()->can('manage-media-meta'),
                    'additional_columns' => [
                        trans('fields.status') => 'parent.node.status'
                    ]
                ]
            );

    }

    /**
     * Store image during upload
     *
     * @param StoreImage $request validated incoming request
     * @param Slider $slider Slider whose image is being uploaded
     * @param Slide $slide empty slide instance
     * @param Node $node empty node instance
     * @return mixed collection of all slides
     */
    public function store(StoreImage $request, Slider $slider, Slide $slide, Node $node)
    {
        // Find the upload path for the product image
        $upload_path = config('media.' . Slide::class . '.upload_path');

        // Save the image in this location
        $path = $request->file('file')->store($upload_path, 'public');

        // Create instance for slide
        $slider->slides()->save($slide);

        $slide->node()->save($node);

        // Create media object
        $media = new Media([
            'src' => $path,
            'title' => $request->file('file')->getClientOriginalName(),
            'type' => $request->file('file')->getMimeType(),
        ]);

        // Save this media file for the product
        $slide->media()->save($media);

        // Add a unique seo friendly name for the image
        $media->url_name =  $media->id . '-' . $request->file('file')->getClientOriginalName();
        $media->save();

        return $this->getAdminSlides($slider);
    }

    /**
     * Edit an existing slide - set content values or change image
     *
     * @param Slider $slider
     * @param Slide $slide
     * @return \Illuminate\View\View
     */
    public function edit(Slider $slider, Slide $slide)
    {
        $slide->media = $slide->media->map(
            function (Media $media) {
                $media->href = asset('storage/' . $media->src);
                $media->can_resize = !in_array($media->type, Media::$raw_image_formats);
                return $media;
            });
        return (new ItemBuilder('slider.admin.slides', $slide, $slide->node))
            ->routeParams([
                $slider->id,
                $slide->id
            ])
            ->view('slideshow::admin.slider.slide', [
                'hide_description' => true,
                'sizes' => $this->getSizeCollection()
            ]);
    }

    /**
     * Update slide content / image
     *
     * @param Request $request incoming request
     * @param Slider $slider Slider whose slide is edited
     * @param Slide $slide current slider
     * @return RedirectResponse|Redirector
     */
    public function update(Request $request, Slider $slider, Slide $slide)
    {
        if ($request->file('slide')) {

            $this->validate($request, [
                'slide' => 'image'
            ]);

            // Find the upload path for the product image
            $upload_path = config('media.' . Slide::class . '.upload_path');

            // Save the image in this location
            $path = $request->file('slide')->store($upload_path, 'public');

            $slide_image = $slide->media()->first();
            $slide_image->src = $path;
            $slide_image->save();

        }
        $slide->node->fill($request->input('node', []));
        $slide->node->save();

        $request->session()->flash('success', trans('slideshow::messages.slide_updated'));
        return redirect(action('\\' . self::class . '@index', [$slider->id]));
    }

    /**
     * Get the size collection for media files
     *
     * @return Collection
     */
    private function getSizeCollection()
    {
        return collect(config('media.' . Slide::class))
            ->reject(function ($entry) {
                // Original upload path is a string, it is not a valid path
                return !is_array($entry);
            })
            ->keyBy(function ($size) {
                return ucfirst(basename($size['path']));
            });
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id Media ID
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Slider $slider, $slide)
    {
        dd($slide);
        $slide->node()->delete();
        $slide->media()->delete();
        $slide->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', trans('slideshow::messages.slide_deleted'));
        return redirect()->back();
    }

    /**
     * Update order of images upon re-order
     *
     * @param Request $request incoming request
     * @param Product $product product whose images are in request
     */
    public function changeOrder(Request $request, Slider $slider)
    {
        foreach ($request->input('media') as $order => $media) {

            Slide::where('id', $media['parent_id'])
                ->update([
                    'order' => $order
                ]);
        }
    }

    /**
     * Prepare slides for admin display
     *
     * @param Slider $slider Current slider
     * @return Collection Prepared slides
     */
    private function getAdminSlides(Slider $slider)
    {
        return $slider->slides->map(
            function (Slide $slide) use ($slider) {
                $media = $slide->media->first();
                $media->load('parent.node');
                $media->href = asset('storage/' . $media->src);
                $media->can_resize = !in_array($media->type, Media::$raw_image_formats);
                $media->edit_url = action('\\' . SlideController::class . '@edit', [
                    $slider->id,
                    $slide->id
                ]);
                return $media;
            }
        );
    }
}