<?php
/**
 * Class Slider
 *
 * @package Mtc\Slideshow
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Slideshow\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Auth;
use \Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Mtc\Core\Media;
use Mtc\Core\Node;
use Mtc\Core\Nodeable;
use Mtc\Slideshow\Http\Controllers\Admin\SlideController;

/**
 * Class Slider
 *
 * Slider model management
 *
 * @package Mtc\Slideshow
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-06-14
 */
class Slider extends Model
{
    use Nodeable;

    /**
     * The attributes to load with this object when instantiating
     *
     * @var array
     */
    protected $with = [
        'node'
    ];

    /**
     * @var string $touches attributes that get updated when model is saved
     */
    protected $touches = [
        'node',
    ];

    /**
     * Relationship with the slides
     *
     * @return HasMany
     */
    public function slides()
    {
        return $this->hasMany(Slide::class)
            ->orderBy('order', 'asc');
    }

    /**
     * Check if slider is viewable
     *
     * @return bool whether slider is viewable
     */
    public function viewable()
    {
        return $this->node->status === Node::NODE_STATUS_PUBLISHED
            && ($this->node->visibility == Node::NODE_VISIBILITY_PUBLIC || Auth::user()->can('admin'));
    }


    /**
     * Build a search query for the admin panel.
     *
     * @param Builder $query Model Query
     * @param Request $request Browser Request
     * @return Builder
     */
    public function scopeBuildSearch(Builder $query, Request $request) {
        if ($request->has('title')) {
            $query->whereHas(
                'node', function ($query) use ($request) {
                $query->where('title', 'LIKE', '%' . $request->input('title') . '%');
            });
        };

        return $query;
    }

    /**
     * Render the Slider.
     *
     * @param integer $id ID of the slider. Optional if multiple_sliders is disabled
     * @return View|void
     */
    public static function render($id = false)
    {
        /** @var self $slider type-hint that slider will be an instance of self if it is found */
        if (!$id && config('slideshow.multiple_sliders')) {
            $slider = self::first();
        } else {
            $slider = self::find($id);
        }

        // Slider not found
        if (!$slider) {
            return;
        }

        // Slider is not viewable
        if (!$slider->viewable()) {
            return;
        }

        // Make sure image is mapped correctly for slider
        $slider->slides = $slider->slides->map(function ($slide) {
            $slide->media = $slide->media->first();
            return $slide;
        });

        // Return the view of the slider
        return view('slideshow::public.slideshow.slider', compact('slider'));
    }
}