<?php
/**
 * Class Feed
 *
 * PHP Version 7
 *
 * @category Mtc\SocialFeed\Google
 * @package  Mtc\SocialFeed\Google
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 * @version 2017-07-12
 */

namespace Mtc\SocialFeed\Google;

use Google_Client;
use Google_Service_Plus;
use Mtc\SocialFeed\SocialFeed;
use Carbon\Carbon;

/**
 * Class Feed
 *
 * @category Mtc\SocialFeed\Google
 * @package  Mtc\SocialFeed\Google
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 * @version 2017-07-12
 */
class Feed
{
    /**
     * @var string
     */
    private $page_id;

    /**
     * @var array
     */
    private $config;

    /**
     * @var Google_Client
     */
    private $client;

    /**
     * @var Google_Service_Plus
     */
    private $google_plus;

    /**
     * Create a new Feed Instance.
     */
    public function __construct()
    {
        $this->page_id = config('socialfeed.google.page_id');

        // Setup and configure the Google Client
        $this->config = ['developer_key' => config('socialfeed.google.access_token')];

        $this->client = new Google_Client($this->config);
        $this->client->setApplicationName('socialfeed');

        $this->google_plus = new Google_Service_Plus($this->client);
    }

    /**
     *  Makes a request to the api and returns the body
     *
     * @return array $data
     */
    public function getFeed()
    {
        $posts = $this->makeApiCall();

        $this->storePosts($posts);

        $this->removePosts();
    }

    /**
     * Process a request at the google api endpoint
     * and return the response as an array
     *
     * @return mixed
     */
    private function makeApiCall()
    {
        return $this->google_plus->activities->listActivities($this->page_id, 'public');
    }

    /**
     * Given an array, store each element into the social_feed table
     *
     * @param array $posts
     * @return void
     */
    private function storePosts($posts)
    {
        foreach ($posts as $post) {

            $post = $this->formatPosts($post);

            $post['updated_at'] = date('Y-m-d H:i:s');

            SocialFeed::updateOrCreate(
                [
                    'post_id' => $post['post_id']
                ],
                $post
            );
        }
    }

    /**
     * Given an array object, define the key/value pair for the array
     * and return back as array
     *
     * @param array $post
     * @return array $tmp_post
     */
    private function formatPosts($post)
    {
        $tmp_post = [
            'permalink' => $post['url'],
            'post_id' => $post['id'],
            'date_posted' => $post['published'],
            'type' => 'googleplus',
            'likes' => $post['object']['plusoners']['totalItems'],
            'message' => isset($post['object']['content']) ?
                $post['object']['content'] : '',
            'image_url' => isset($post['object']['attachments'][0]['fullImage']['url']) ?
                $post['object']['attachments'][0]['fullImage']['url'] : '',
        ];

        return $tmp_post;
    }

    /**
     * Find all post tagged as type googleplus and remove any
     * not updated in the last two weeks
     *
     * @return void
     */
    private function removePosts()
    {
        $now = new Carbon();

        SocialFeed::where('type', '=', 'googleplus')
            ->where('updated_at', '<=', $now->subWeeks(2)->toDateString())
            ->delete();
    }
}
