<?php
/**
 * Feed Class
 *
 * PHP Version 7
 *
 * @category Mtc\SocialFeed\Instagram
 * @package  Mtc\SocialFeed\Instagram
 * @author   Anthony Grimes <anthony.grimes@mtcmedia.co.uk>
 * @version 2017-07-12
 */

namespace Mtc\SocialFeed\Instagram;

use Mtc\SocialFeed\SocialFeed;
use Carbon\Carbon;

/**
 * Feed Class
 *
 * @category Mtc\SocialFeed\Instagram
 * @package  Mtc\SocialFeed\Instagram
 * @author   Anthony Grimes <anthony.grimes@mtcmedia.co.uk>
 * @version 2017-07-12
 */
class Feed
{
    /**
     * @var string
     */
    private $instagram_access_token;

    /**
     * Feed constructor.
     */
    public function __construct()
    {
        $this->instagram_access_token = config('socialfeed.instagram.access_token');
    }

    /**
     * Retrieves a user feed, stores the request and performs database maintenance
     *
     * @return void
     */
    public function getFeed()
    {
        $posts = $this->makeApiCall();

        $this->storePosts($posts);

        $this->removePosts();
    }

    /**
     * Process a request at the instagram api endpoint
     * and return the response as an array
     *
     * @return array
     */
    private function makeApiCall()
    {
        $api_call = 'https://api.instagram.com/v1/users/self/media/recent/?access_token=' . $this->instagram_access_token;

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $api_call);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Accept: application/json'));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

        $data = curl_exec($ch);

        curl_close($ch);

        return json_decode($data, true);
    }

    /**
     * Given an array, store each element into the social_feed table
     *
     * @param array $posts
     * @return void
     */
    private function storePosts($posts)
    {
        foreach ($posts['data'] as $post) {

            $post = $this->formatPosts($post);

            $post['updated_at'] = date('Y-m-d H:i:s');

            SocialFeed::updateOrCreate(
                [
                    'post_id' => $post['post_id']
                ],
                $post
            );
        }
    }

    /**
     * Given an array object, define the key/value pair for the array
     * and return back as array
     *
     * @param array $post
     * @return array $tmp_post
     */
    private function formatPosts($post)
    {
        $tmp_post = [
            'message' => !empty($post['caption']['text']) ? $post['caption']['text'] : '',
            'likes' => $post['likes']['count'],
            'image_url' => $post['images']['standard_resolution']['url'],
            'permalink' => $post['link'],
            'date_posted' => date('Y-m-d H:i:s', $post['caption']['created_time']),
            'type' => 'instagram',
            'post_id' => $post['id']
        ];

        return $tmp_post;
    }

    /**
     * Find all post tagged as type instagram and remove any
     * not updated in the last two weeks
     *
     * @return void
     */
    private function removePosts()
    {
        $now = new Carbon();

        SocialFeed::where('type', '=', 'instagram')
            ->where('updated_at', '<=', $now->subWeeks(2)->toDateString())
            ->delete();
    }
}
