<?php
/**
 * Class Feed
 *
 * PHP Version 7
 *
 * @category Mtc\SocialFeed\Facebook
 * @package  Mtc\SocialFeed\Facebook
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 * @version 2017-07-12
 */

namespace Mtc\SocialFeed\Facebook;

use Carbon\Carbon;
use Facebook\Exceptions\FacebookSDKException;
use Facebook\Facebook;
use Mtc\SocialFeed\SocialFeed;

/**
 * Class Feed
 *
 * @category Mtc\SocialFeed\Facebook
 * @package  Mtc\SocialFeed\Facebook
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 * @version 2017-07-12
 */
class Feed
{

    /**
     * @var Facebook
     */
    private $fb;
    /**
     * @var array
     */
    private $fields;
    /**
     * @var array
     */
    private $reactions;

    /**
     * Create a new Feed Instance.
     */
    public function __construct()
    {
        $this->fb = new Facebook([
            'app_id' => config('socialfeed.facebook.app_id'),
            'app_secret' => config('socialfeed.facebook.app_secret'),
            'default_graph_version' => 'v2.9',
        ]);
        //fields requested from the API
        $this->fields = [
            'permalink_url',
            'id',
            'created_time',
            'type',
            'message',
            'full_picture'
        ];
        // types of reactions to include for counting likes
        $this->reactions = [
            'like',
            'love',
            'wow',
            'haha',
            'sad',
            'angry',
            'thankful'
        ];
    }

    /**
     * Retrieves a user feed, stores the request and performs database maintenance
     *
     * @return void
     */
    public function getFeed()
    {
        $posts = $this->makeApiCall();
        $this->storePosts($posts);
        $this->removePosts();
    }

    /**
     * Makes a request to the api and returns the decoded body
     *
     * @return array
     */
    private function makeApiCall()
    {
        $response = $this->fb->get('/' . config('socialfeed.facebook.page_id') . '/posts?fields=' .
        $this->buildRequest(),
        $this->token());
        
        return $data = $response->getDecodedBody()['data'];
    }

    /**
     *  Builds the request from the fields defined in the $fields variable
     *
     * @return string
     */
    private function buildRequest()
    {
        $request_string = '';
        // foreach requested field
        foreach ($this->fields as $field) {
            $request_string .= $field . ',';
        }
        // and each requested reaction
        foreach ($this->reactions as $reaction) {
            $request_string .= 'reactions.type(' . strtoupper($reaction) .
                ').limit(0).summary(1).as(' . $reaction . '),';
        }
        // return the request string minus the trailing comma
        return substr($request_string, 0, -1);
    }

    /**
     *  Returns the access token needed to make requests to the API
     *
     * @return string
     */
    private function token()
    {
        return (config('socialfeed.facebook.app_id') . '|' . config('socialfeed.facebook.app_secret'));
    }

    /**
     *  Saves the posts to the database
     *
     * @param array $data decoded body of posts returned from the API
     * @return void
     */
    public function storePosts($posts)
    {
        foreach ($posts as $post) {
            $post = $this->formatPosts($post);

            $post['updated_at'] = date('Y-m-d H:i:s');

            SocialFeed::updateOrCreate(
                [
                    'post_id' => $post['post_id']
                ],
                $post
            );
        }
    }

    /**
     * Given an array object, define the key/value pair for the array
     * and return back as array
     *
     * @param array $post
     * @return array $tmp_post
     */
    private function formatPosts($post)
    {
        $tmp_post = [
            'permalink' => $post['permalink_url'],
            'post_id' => $post['id'],
            'date_posted' => Carbon::createFromFormat('Y-m-d\TH:i:sO', $post['created_time']),
            'type' => 'facebook',
            'likes' => $this->getInteractions($post),
            'message' => isset($post['message']) ? $post['message'] : '',
            'image_url' => isset($post['full_picture']) ? $post['full_picture'] : ''
        ];

        return $tmp_post;
    }

    /**
     *  Counts the number of likes and reactions ('interactions') on a post
     *
     * @param array $post
     * @return int
     */
    private function getInteractions($post)
    {
        $interactions = 0;
        foreach ($this->reactions as $reaction) {
            $interactions += $post[$reaction]['summary']['total_count'];
        }
        return $interactions;
    }


    /**
     * Find all post tagged as type facebook and remove any
     * not updated in the last two weeks
     *
     * @return void
     */
    private function removePosts()
    {
        $now = new Carbon();

        SocialFeed::where('type', '=', 'facebook')
            ->where('updated_at', '<=', $now->subWeeks(2)->toDateString())
            ->delete();
    }
}
