<?php
/**
 * Feed Class
 *
 * PHP Version 7
 *
 * @category Mtc\SocialFeed\Twitter
 * @package  Mtc\SocialFeed\Twitter
 * @author   Anthony Grimes <anthony.grimes@mtcmedia.co.uk>
 * @version 2017-07-12
 */

namespace Mtc\SocialFeed\Twitter;

use Abraham\TwitterOAuth\TwitterOAuth;
use Mtc\SocialFeed\SocialFeed;
use Carbon\Carbon;

/**
 * Feed Class
 *
 * @category Mtc\SocialFeed\Twitter
 * @package  Mtc\SocialFeed\Twitter
 * @author   Anthony Grimes <anthony.grimes@mtcmedia.co.uk>
 * @version 2017-07-12
 */
class Feed
{
    /**
     * @var string
     */
    private $twitter_id;

    /**
     * @var string
     */
    private $twitter_consumer_key;

    /**
     * @var string
     */
    private $twitter_consumer_secret;

    /**
     * @var string
     */
    private $twitter_access_token;

    /**
     * @var string
     */
    private $twitter_access_token_secret;

    /**
     * @var TwitterOAuth
     */
    private $connection;

    /**
     * Feed constructor.
     */
    public function __construct()
    {
        $this->twitter_id = config('socialfeed.twitter.id');

        $this->twitter_consumer_key = config('socialfeed.twitter.consumer_key');
        $this->twitter_consumer_secret = config('socialfeed.twitter.consumer_secret');
        $this->twitter_access_token = config('socialfeed.twitter.access_token');
        $this->twitter_access_token_secret = config('socialfeed.twitter.access_token_secret');

        $this->connection = new TwitterOAuth($this->twitter_consumer_key, $this->twitter_consumer_secret,
            $this->twitter_access_token, $this->twitter_access_token_secret);
    }

    /**
     * Retrieves a user feed, stores the request and performs database maintenance
     *
     * @return void
     */
    public function getFeed()
    {
        $tweets = $this->makeApiCall();

        $this->storeTweets($tweets);

        $this->removeTweets();
    }

    /**
     * Process a request at the twitter api endpoint
     * and return the response as an array
     *
     * @return array
     */
    private function makeApiCall()
    {
        $api_endpoint = "statuses/user_timeline";

        $api_params = [
            "user_id" => $this->twitter_id,
            "count" => 50,
            "exclude_replies" => true,
            "include_rts" => false
        ];

        $data = $this->connection->get($api_endpoint, $api_params);

        $data = json_decode(json_encode($data), true);

        return $data;
    }

    /**
     * Given an array, store each element into the social_feed table
     *
     * @param array $tweets
     * @return void
     */
    private function storeTweets($tweets)
    {
        foreach ($tweets as $tweet) {

            $tweet = $this->formatTweets($tweet);

            $tweet['updated_at'] = date('Y-m-d H:i:s');

            SocialFeed::updateOrCreate(
                [
                    'post_id' => $tweet['post_id']
                ],
                $tweet
            );
        }
    }

    /**
     * Given an array object, define the key/value pair for the array
     * and return back as array
     *
     * @param array $tweet
     * @return array $tmp_tweet
     */
    private function formatTweets($tweet)
    {
        $tmp_tweet = [
            'message' => $tweet['text'],
            'likes' => $tweet['favorite_count'],
            'image_url' => !empty($tweet['entities']['media'][0]['media_url']) ? $tweet['entities']['media'][0]['media_url'] : '',
            'permalink' => 'https://www.twitter.com/' . $tweet['user']['screen_name'] . '/status/' . $tweet['id_str'],
            'date_posted' => date('Y-m-d H:i:s', strtotime($tweet['created_at'])),
            'type' => 'twitter',
            'post_id' => $tweet['id_str']
        ];

        return $tmp_tweet;
    }

    /**
     * Find all post tagged as type twitter and remove any
     * not updated in the last two weeks
     *
     * @return void
     */
    private function removeTweets()
    {
        $now = new Carbon();

        SocialFeed::where('type', '=', 'twitter')
            ->where('created_at', '<=', $now->subWeeks(2)->toDateString())
            ->delete();
    }
}
