<?php

namespace Mtc\Components\Webhooks\Jobs;

use GuzzleHttp\Client;
use GuzzleHttp\Middleware;
use GuzzleHttp\Promise\PromiseInterface;
use Illuminate\Bus\Queueable;
use Mtc\Components\Webhooks\WebhookLog;
use Illuminate\Queue\SerializesModels;
use Mtc\Components\Webhooks\Webhook;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;

/**
 * Class RunWebhook
 * Runs a webhook job
 *
 * @package Mtc\Components\Webhooks\Jobs
 */
class RunWebhook implements ShouldQueue
{
    use Queueable, InteractsWithQueue, SerializesModels;

    /**
     * Webhook that needs to be run
     *
     * @var Webhook
     */
    protected $webhook;

    /**
     * The event data to be posted to our hooks.
     *
     * @var mixed
     */
    protected $event_data;

    /**
     * Create a new job instance.
     *
     * @param Webhook $webhooks
     * @param mixed $eventData
     */
    public function __construct($webhook, $event_data)
    {
        $this->event_data = $event_data;
        $this->webhook = $webhook;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // skip if we're over the limit of max attempts
        if (config('webhooks.log.max_attempts', -1) !== -1 && $this->attempts() > config('webhooks.log.max_attempts')) {
            return;
        }

        $this->event_data->event = $this->webhook->event_name;

        $client = app(Client::class);
        if (config('webhooks.log.active')) {

            // Clear previous entry if needed
            if ($this->shouldClearLog()) {
                $this->webhook
                    ->logs()
                    ->orderBy('updated_at', 'asc')
                    ->first()
                    ->delete();
            }

            // Set up Logger instance
            $log = new WebhookLog([
                'webhook_id' => $this->webhook['id'],
                'url' => $this->webhook['url'],
            ]);

            $middleware = Middleware::tap(function (RequestInterface $request, $options) use ($log) {
                $log->payload_format = isset($request->getHeader('Content-Type')[0]) ? $request->getHeader('Content-Type')[0] : null;
                $log->payload = $request->getBody()->getContents();
            }, function ($request, $options, PromiseInterface $response) use ($log) {
                $response->then(function (ResponseInterface $response) use ($log) {
                    $log->status = $response->getStatusCode();
                    $log->response = $response->getBody()->getContents();
                    $log->response_format = $log->payload_format = isset($response->getHeader('Content-Type')[0]) ? $response->getHeader('Content-Type')[0] : null;
                    $log->save();

                    // Retry this job if the webhook response didn't give us a HTTP 200 OK
                    if ($response->getStatusCode() >= 300 || $response->getStatusCode() < 200) {
                        $this->release(30);
                    }

                });
            });

            $client->post($this->webhook['url'], [
                'exceptions' => false,
                'body' => json_encode($this->event_data),
                'verify' => false,
                'handler' => $middleware($client->getConfig('handler')),
                'timeout' => 3,
            ]);

            return;
        }

        // Here we don't have logging so we have have a simplified request with no concern about response
        $client->post($this->webhook['url'], [
            'exceptions' => false,
            'body' => json_encode($this->event_data),
            'verify' => false,
            'timeout' => 3,
        ]);
    }

    /**
     * Check if last entry for the log needs to be cleared
     *
     * @return bool
     */
    protected function shouldClearLog()
    {
        return config('webhooks.log.storage_quantity') > 0 && $this->webhook->logs()->count() >= config('webhooks.log.log_length_per_hook');
    }
}
